#import "ViewController.h"
#import "GLProgram.h"

typedef struct {
    GLKVector3 position;
} Vertex;

const Vertex SquareVertices[] = {
    {-1, -1 , 0}, // bottom left
    {1, -1 , 0},  // bottom right
    {1, 1 , 0},   // top right
    {-1, 1 , 0},  // top left
};

const GLubyte SquareTriangles[] = {
    0, 1, 2, // BL -> BR -> TR
    2, 3, 0  // TR -> TL -> BL
};

@interface ViewController () {
    GLuint _vertexBuffer; // contains the collection of vertices used to describe position of each corner
    GLuint _indexBuffer;  // indicates which vertices should be used in each triangle used to make up the square
    GLProgram* _program;
    GLint uMVPMatrix;
    GLKMatrix4 _modelViewProjectionMatrix;
}
@end

@implementation ViewController

- (void)viewDidLoad
{
    [super viewDidLoad];
    
    GLKView* view = (GLKView*)self.view;
    view.context = [[EAGLContext alloc] initWithAPI:kEAGLRenderingAPIOpenGLES2];
    
    [EAGLContext setCurrentContext:view.context];
    
    _program = [[GLProgram alloc] initWithVertexShader:SIMPLE_VERTEX FragmentShader:SIMPLE_FRAGMENT];
    
    if (![_program link])
    {
        NSString *programLog = [_program programLog];
        NSLog(@"Program link log: %@", programLog);
        NSString *fragmentLog = [_program fragmentShaderLog];
        NSLog(@"Fragment shader compile log: %@", fragmentLog);
        NSString *vertexLog = [_program vertexShaderLog];
        NSLog(@"Vertex shader compile log: %@", vertexLog);
        _program = nil;
        NSAssert(NO, @"Falied to link HalfSpherical shaders");
    }
    [_program addAttribute:@"position"];
    uMVPMatrix = [_program uniformIndex:@"modelViewProjectionMatrix"];
    
    glGenBuffers(1, &_vertexBuffer);
    glBindBuffer(GL_ARRAY_BUFFER, _vertexBuffer);
    glBufferData(GL_ARRAY_BUFFER, sizeof(SquareVertices), SquareVertices, GL_STATIC_DRAW);
    glEnableVertexAttribArray(GLKVertexAttribPosition);
    glVertexAttribPointer(GLKVertexAttribPosition, 3, GL_FLOAT, GL_FALSE, sizeof(GLfloat) * 3, NULL);
    
    glGenBuffers(1, &_indexBuffer);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, _indexBuffer);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(SquareTriangles), SquareTriangles, GL_STATIC_DRAW);
 
    float aspectRatio = self.view.bounds.size.width / self.view.bounds.size.height;
    float fieldOfViewDegrees = 60.0;
    GLKMatrix4 projectionMatrix = GLKMatrix4MakePerspective(GLKMathDegreesToRadians(fieldOfViewDegrees), aspectRatio, 0.1, 10.0);
    GLKMatrix4 modelViewMatrix = GLKMatrix4MakeTranslation(0.0f, 0.0f, -6.0f);
    _modelViewProjectionMatrix = GLKMatrix4Multiply(projectionMatrix, modelViewMatrix);
    
    [_program use];
}

- (void)glkView:(GLKView *)view drawInRect:(CGRect)rect {
    glClearColor(0.0, 0.0, 0.0, 1.0);
    glClear(GL_COLOR_BUFFER_BIT);
    
    glUniformMatrix4fv(uMVPMatrix, 1, 0, _modelViewProjectionMatrix.m);
    
    // Now that OpenGL knows where to find vertex positions, it can draw them.
    int numberOfVertices = sizeof(SquareTriangles)/sizeof(SquareTriangles[0]);
    glDrawElements(GL_TRIANGLES, numberOfVertices, GL_UNSIGNED_BYTE, 0);
}

- (void)didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
    // Dispose of any resources that can be recreated.
}

@end
